/*
 * Decompiled with CFR 0.152.
 */
package com.craftjakob.configapi.config;

import com.craftjakob.configapi.config.Config;
import com.craftjakob.configapi.config.ConfigData;
import com.craftjakob.configapi.config.ConfigRegister;
import com.craftjakob.platform.Platform;
import com.mojang.logging.LogUtils;
import java.io.IOException;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.util.Collection;
import java.util.EnumMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.stream.Collectors;
import net.minecraft.server.MinecraftServer;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;

public class ConfigTracker {
    private static final ConfigTracker INSTANCE = new ConfigTracker();
    private static final Logger LOGGER = LogUtils.getLogger();
    private final EnumMap<Config.ConfigType, Set<Config>> configSets = ConfigRegister.getConfigSets();
    private final ConcurrentHashMap<String, Map<Config.ConfigType, LinkedHashSet<Config>>> configsByModId = new ConcurrentHashMap();
    private final ConfigWatcher configWatcher = new ConfigWatcher().startWatching();
    private boolean areServerConfigsLoaded = false;

    private ConfigTracker() {
    }

    public static ConfigTracker get() {
        return INSTANCE;
    }

    public synchronized void trackConfig(Config config) {
        Map modConfigs = this.configsByModId.computeIfAbsent(config.getModId(), k -> new EnumMap(Config.ConfigType.class));
        LinkedHashSet configs = modConfigs.computeIfAbsent(config.getType(), k -> new LinkedHashSet());
        configs.add(config);
        if (config.getType() == Config.ConfigType.COMMON) {
            this.loadConfig(config);
        }
    }

    public void loadClientConfigs() {
        this.loadConfigsForType(Config.ConfigType.CLIENT);
    }

    private void loadConfigsForType(Config.ConfigType type) {
        this.configSets.get((Object)type).forEach(this::loadConfig);
    }

    public void loadConfig(Config config) {
        if (config.getType() != Config.ConfigType.SERVER) {
            config.setConfigFile(Platform.getConfigPath());
        }
        config.load();
        config.save();
        if (config.isTracked()) {
            this.configWatcher.addConfig(config);
        }
    }

    public void loadServerConfigs(Path serverConfigDirectory) {
        if (this.areServerConfigsLoaded) {
            return;
        }
        this.configSets.get((Object)Config.ConfigType.SERVER).forEach(config -> {
            config.setConfigFile(serverConfigDirectory.toAbsolutePath());
            if (config.getConfigData() == null) {
                config.setConfigData(config.configureConfigData(config.getConfigurator()));
            }
            this.loadConfig((Config)config);
            if (config.isTracked()) {
                this.configWatcher.addConfig((Config)config);
            }
        });
        this.areServerConfigsLoaded = true;
    }

    public void loadServerConfigs(MinecraftServer server) {
        this.loadServerConfigs(Platform.getServerConfigPath(server));
    }

    public void unloadConfigs(Config.ConfigType type) {
        if (type == Config.ConfigType.SERVER && !this.areServerConfigsLoaded) {
            return;
        }
        this.configSets.get((Object)type).forEach(config -> {
            if (config.isTracked()) {
                this.stopTrackingConfig((Config)config);
            }
            config.clearServerConfigData();
        });
        if (type == Config.ConfigType.SERVER) {
            this.areServerConfigsLoaded = false;
        }
    }

    public void reload(Config.ConfigType type) {
        this.configSets.get((Object)type).forEach(config -> {
            config.load();
            config.save();
        });
    }

    public void stopTrackingConfig(Config config) {
        if (this.configWatcher.removeConfig(config)) {
            LOGGER.info("The tracking of config file " + config.getFilePath() + " was stopped.");
        }
    }

    public void stopTrackingConfigs() {
        this.configWatcher.stopWatching();
    }

    public void startTrackingConfigs() {
        this.configWatcher.startWatching();
    }

    public ConcurrentHashMap<String, Map<Config.ConfigType, LinkedHashSet<Config>>> getConfigsByModId() {
        return this.configsByModId;
    }

    public void acceptSyncedConfigs(String fileName, byte[] data) {
        for (Config config : this.configWatcher.getConfigs()) {
            if (!config.getFilePath().equals(fileName)) continue;
            String configData = new String(data);
            ConfigData newConfigData = config.getConfigData().parseConfigData(configData);
            config.setConfigData(newConfigData);
            break;
        }
    }

    public Map<String, byte[]> getConfigSync() {
        return this.configSets.get((Object)Config.ConfigType.SERVER).stream().collect(Collectors.toMap(Config::getFilePath, file -> {
            try {
                return Files.readAllBytes(file.getConfigFile().toPath());
            }
            catch (IOException e) {
                throw new RuntimeException("Failed syncing config! Config file " + String.valueOf(file) + " does not exist! This can happen, if the server tries to get the config file data before the file is created. Just join again.", e);
            }
        }));
    }

    public static class ConfigWatcher
    implements Runnable {
        private final WatchService watchService;
        private final ExecutorService executorService;
        private final ConcurrentHashMap<Path, Pair<Config, WatchKey>> configMap;

        public ConfigWatcher() {
            try {
                this.watchService = FileSystems.getDefault().newWatchService();
            }
            catch (IOException e) {
                throw new RuntimeException("Failed to create WatchService", e);
            }
            this.executorService = Executors.newSingleThreadExecutor();
            this.configMap = new ConcurrentHashMap();
        }

        public void addConfig(Config config) {
            Path configPath = config.getConfigFile().toPath();
            try {
                WatchKey watchKey = configPath.getParent().register(this.watchService, StandardWatchEventKinds.ENTRY_MODIFY);
                this.configMap.put(configPath, (Pair<Config, WatchKey>)Pair.of((Object)config, (Object)watchKey));
            }
            catch (IOException e) {
                throw new RuntimeException("Failed to register path: " + String.valueOf(configPath.getParent()) + " for config file: " + config.getFilePath(), e);
            }
        }

        public boolean removeConfig(Config config) {
            Path configPath = config.getConfigFile().toPath();
            Pair<Config, WatchKey> pair = this.configMap.remove(configPath);
            if (pair != null) {
                ((WatchKey)pair.getRight()).cancel();
                return true;
            }
            return false;
        }

        public ConfigWatcher startWatching() {
            this.executorService.submit(this);
            return this;
        }

        public void stopWatching() {
            this.executorService.shutdownNow();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            try {
                WatchKey key;
                while ((key = this.watchService.take()) != null) {
                    for (WatchEvent<?> event : key.pollEvents()) {
                        if (event.kind() != StandardWatchEventKinds.ENTRY_MODIFY) continue;
                        Path changed = (Path)event.context();
                        Path absolutePath = ((Path)key.watchable()).resolve(changed).toAbsolutePath();
                        if (!this.configMap.containsKey(absolutePath)) continue;
                        Config config = (Config)this.configMap.get(absolutePath).getLeft();
                        if (config != null) {
                            config.load();
                            LOGGER.info("Config file " + config.getFilePath() + " changed, reloading config...");
                            continue;
                        }
                        LOGGER.error("Failed to get Config, because of changed path: " + String.valueOf(changed) + " absolute path: " + String.valueOf(absolutePath));
                    }
                    key.reset();
                }
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                LOGGER.error("Watcher thread interrupted", (Throwable)e);
            }
            finally {
                try {
                    this.watchService.close();
                }
                catch (IOException e) {
                    LOGGER.error("Failed to close WatchService", (Throwable)e);
                }
            }
        }

        public Collection<Config> getConfigs() {
            return this.configMap.values().stream().map(Pair::getLeft).toList();
        }
    }
}

